/*
 * Program:	SSL authentication/encryption module
 *
 * Author:	Mark Crispin
 *		Networks and Distributed Computing
 *		Computing & Communications
 *		University of Washington
 *		Administration Building, AG-44
 *		Seattle, WA  98195
 *		Internet: MRC@CAC.Washington.EDU
 *
 * Date:	22 September 1998
 * Last Edited:	12 October 1999
 *
 * Copyright 1999 by the University of Washington
 *
 * This software is provided under specific, written license from the
 * University of Washington and may only be used, copied, modified, and
 * distributed under the terms of such license.  This software is made
 * available "as is", and
 * THE UNIVERSITY OF WASHINGTON DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN
 * NO EVENT SHALL THE UNIVERSITY OF WASHINGTON BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT
 * (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Export Regulations. Software, including technical data, is subject to U.S.
 * export control laws, including the U.S. Export Administration Act and its
 * associated regulations, and may be subject to export or import regulations
 * in other countries. Licensee agrees to comply strictly with all such
 * regulations and acknowledges that it has the responsibility to obtain
 * licenses to export, re-export, or import Software. Software may not be
 * downloaded, or otherwise exported or re-exported (i) into, or to a
 * national or resident of, Cuba, Iraq, Iran, North Korea, Libya, Sudan,
 * Syria or any country to which the U.S. has embargoed goods; or (ii) to
 * anyone on the U.S. Treasury Department's list of Specially Designated
 * Nations or the U.S. Commerce Department's Table of Denial Orders.
 */

#define crypt ssl_private_crypt
#include <x509.h>
#include <ssl.h>
#include <err.h>
#include <pem.h>
#include <buffer.h>
#include <bio.h>
#include <crypto.h>
#undef crypt

#define SSLBUFLEN 8192

#ifndef SSL_CERT_DIRECTORY
#define SSL_CERT_DIRECTORY "/usr/local/ssl/certs/"
#endif


/* SSL I/O stream */

typedef struct ssl_stream {
  TCPSTREAM *tcpstream;		/* TCP stream */
  SSL_CTX *context;		/* SSL context */
  SSL *con;			/* SSL connection */
  int ictr;			/* input counter */
  char *iptr;			/* input pointer */
  char ibuf[SSLBUFLEN];		/* input buffer */
} SSLSTREAM;


/* SSL stdio stream */

typedef struct ssl_stdiostream {
  SSLSTREAM *sslstream;		/* SSL stream */
  int octr;			/* output counter */
  char *optr;			/* output pointer */
  char obuf[SSLBUFLEN];		/* output buffer */
} SSLSTDIOSTREAM;


/* SSL driver */

struct ssl_driver {		/* must parallel NETDRIVER in mail.h */
  SSLSTREAM *(*open) (char *host,char *service,unsigned long port);
  SSLSTREAM *(*aopen) (NETMBX *mb,char *service,char *usrbuf);
  char *(*getline) (SSLSTREAM *stream);
  long (*getbuffer) (SSLSTREAM *stream,unsigned long size,char *buffer);
  long (*soutr) (SSLSTREAM *stream,char *string);
  long (*sout) (SSLSTREAM *stream,char *string,unsigned long size);
  void (*close) (SSLSTREAM *stream);
  char *(*host) (SSLSTREAM *stream);
  char *(*remotehost) (SSLSTREAM *stream);
  unsigned long (*port) (SSLSTREAM *stream);
  char *(*localhost) (SSLSTREAM *stream);
};

/* Function prototypes */

void ssl_onceonlyinit (void);
SSLSTREAM *ssl_open (char *host,char *service,unsigned long port);
SSLSTREAM *ssl_aopen (NETMBX *mb,char *service,char *usrbuf);
char *ssl_getline (SSLSTREAM *stream);
long ssl_getbuffer (SSLSTREAM *stream,unsigned long size,char *buffer);
long ssl_getdata (SSLSTREAM *stream);
long ssl_soutr (SSLSTREAM *stream,char *string);
long ssl_sout (SSLSTREAM *stream,char *string,unsigned long size);
void ssl_close (SSLSTREAM *stream);
long ssl_abort (SSLSTREAM *stream);
char *ssl_host (SSLSTREAM *stream);
char *ssl_remotehost (SSLSTREAM *stream);
unsigned long ssl_port (SSLSTREAM *stream);
char *ssl_localhost (SSLSTREAM *stream);
long auth_plain_valid (void);
long auth_plain_client (authchallenge_t challenger,authrespond_t responder,
			NETMBX *mb,void *stream,unsigned long *trial,
			char *user);
char *auth_plain_server (authresponse_t responder,int argc,char *argv[]);
void Server_init (char *server,char *service,char *altservice,char *sasl,
                  void *clkint,void *kodint,void *hupint,void *trmint);
long Server_input_wait (long seconds);
SSLSTDIOSTREAM *ssl_server_init (char *server);
static RSA *ssl_genkey (SSL *con,int export,int keylength);
int ssl_getchar (void);
char *ssl_gets (char *s,int n);
int ssl_putchar (int c);
int ssl_puts (char *s);
int ssl_flush (void);

/* Secure Sockets Layer network driver dispatch */

static struct ssl_driver ssldriver = {
  ssl_open,			/* open connection */
  ssl_aopen,			/* open preauthenticated connection */
  ssl_getline,			/* get a line */
  ssl_getbuffer,		/* get a buffer */
  ssl_soutr,			/* output pushed data */
  ssl_sout,			/* output string */
  ssl_close,			/* close connection */
  ssl_host,			/* return host name */
  ssl_remotehost,		/* return remote host name */
  ssl_port,			/* return port number */
  ssl_localhost			/* return local host name */
};

				/* non-NIL if doing SSL primary I/O */
static SSLSTDIOSTREAM *sslstdio = NIL;


/* Secure sockets layer authenticator */

AUTHENTICATOR auth_ssl = {
  NIL,				/* insecure authenticator */
  "PLAIN",			/* authenticator name */
  auth_plain_valid,		/* check if valid */
  auth_plain_client,		/* client method */
  auth_plain_server,		/* server method */
  NIL				/* next authenticator */
};


/* One-time SSL initialization */

static int sslonceonly = 0;

void ssl_onceonlyinit (void)
{
  if (!sslonceonly++) {		/* only need to call it once */
    mail_parameters (NIL,SET_ALTDRIVER,(void *) &ssldriver);
    mail_parameters (NIL,SET_ALTDRIVERNAME,(void *) "ssl");
    mail_parameters (NIL,SET_ALTIMAPNAME,(void *) "*imaps");
    mail_parameters (NIL,SET_ALTIMAPPORT,(void *) 993);
    mail_parameters (NIL,SET_ALTPOPNAME,(void *) "*pop3s");
    mail_parameters (NIL,SET_ALTPOPPORT,(void *) 995);
    mail_parameters (NIL,SET_ALTNNTPNAME,(void *) "*nntps");
    mail_parameters (NIL,SET_ALTNNTPPORT,(void *) 563);
    mail_parameters (NIL,SET_ALTSMTPNAME,(void *) "*smtps");
    mail_parameters (NIL,SET_ALTSMTPPORT,(void *) 465);
    SSLeay_add_ssl_algorithms ();
  }
}

/* SSL open
 * Accepts: host name
 *	    contact service name
 *	    contact port number
 * Returns: SSL stream if success else NIL
 */

SSLSTREAM *ssl_open (char *host,char *service,unsigned long port)
{
  SSLSTREAM *stream = NIL;
  TCPSTREAM *ts = tcp_open (host,service,port);
  if (ts) {			/* got a TCPSTREAM? */
				/* instantiate SSLSTREAM */
    (stream = (SSLSTREAM *) memset (fs_get (sizeof (SSLSTREAM)),0,
				    sizeof (SSLSTREAM)))->tcpstream = ts;
    if (stream->context = SSL_CTX_new (SSLv23_client_method ())) {
      BIO *bio = BIO_new_socket (ts->tcpsi,BIO_NOCLOSE);
      SSL_CTX_set_options (stream->context,0);
      SSL_CTX_set_verify (stream->context,SSL_VERIFY_NONE,NULL);
      SSL_CTX_set_default_verify_paths (stream->context);
      if (stream->con = (SSL *) SSL_new (stream->context)) {
	SSL_set_bio (stream->con,bio,bio);
	SSL_set_connect_state (stream->con);
	if (SSL_in_init (stream->con)) SSL_total_renegotiations (stream->con);
	return stream;
      }
    }
    ssl_close (stream);		/* failed to do SSL */
  }
  return NIL;
}

  
/* SSL authenticated open
 * Accepts: host name
 *	    service name
 *	    returned user name buffer
 * Returns: SSL stream if success else NIL
 */

SSLSTREAM *ssl_aopen (NETMBX *mb,char *service,char *usrbuf)
{
  return NIL;			/* don't use this mechanism with SSL */
}

/* SSL receive line
 * Accepts: SSL stream
 * Returns: text line string or NIL if failure
 */

char *ssl_getline (SSLSTREAM *stream)
{
  int n,m;
  char *st,*ret,*stp;
  char c = '\0';
  char d;
				/* make sure have data */
  if (!ssl_getdata (stream)) return NIL;
  st = stream->iptr;		/* save start of string */
  n = 0;			/* init string count */
  while (stream->ictr--) {	/* look for end of line */
    d = *stream->iptr++;	/* slurp another character */
    if ((c == '\015') && (d == '\012')) {
      ret = (char *) fs_get (n--);
      memcpy (ret,st,n);	/* copy into a free storage string */
      ret[n] = '\0';		/* tie off string with null */
      return ret;
    }
    n++;			/* count another character searched */
    c = d;			/* remember previous character */
  }
				/* copy partial string from buffer */
  memcpy ((ret = stp = (char *) fs_get (n)),st,n);
				/* get more data from the net */
  if (!ssl_getdata (stream)) fs_give ((void **) &ret);
				/* special case of newline broken by buffer */
  else if ((c == '\015') && (*stream->iptr == '\012')) {
    stream->iptr++;		/* eat the line feed */
    stream->ictr--;
    ret[n - 1] = '\0';		/* tie off string with null */
  }
				/* else recurse to get remainder */
  else if (st = ssl_getline (stream)) {
    ret = (char *) fs_get (n + 1 + (m = strlen (st)));
    memcpy (ret,stp,n);		/* copy first part */
    memcpy (ret + n,st,m);	/* and second part */
    fs_give ((void **) &stp);	/* flush first part */
    fs_give ((void **) &st);	/* flush second part */
    ret[n + m] = '\0';		/* tie off string with null */
  }
  return ret;
}

/* SSL receive buffer
 * Accepts: SSL stream
 *	    size in bytes
 *	    buffer to read into
 * Returns: T if success, NIL otherwise
 */

long ssl_getbuffer (SSLSTREAM *stream,unsigned long size,char *buffer)
{
  unsigned long n;
  while (size > 0) {		/* until request satisfied */
    if (!ssl_getdata (stream)) return NIL;
    n = min (size,stream->ictr);/* number of bytes to transfer */
				/* do the copy */
    memcpy (buffer,stream->iptr,n);
    buffer += n;		/* update pointer */
    stream->iptr += n;
    size -= n;			/* update # of bytes to do */
    stream->ictr -= n;
  }
  buffer[0] = '\0';		/* tie off string */
  return T;
}


/* SSL receive data
 * Accepts: TCP/IP stream
 * Returns: T if success, NIL otherwise
 */

long ssl_getdata (SSLSTREAM *stream)
{
  int i,sock;
  fd_set fds,efds;
  struct timeval tmo;
  tcptimeout_t tmoh = (tcptimeout_t) mail_parameters (NIL,GET_TIMEOUT,NIL);
  long ttmo_read = (long) mail_parameters (NIL,GET_READTIMEOUT,NIL);
  time_t t = time (0);
  blocknotify_t bn = (blocknotify_t) mail_parameters (NIL,GET_BLOCKNOTIFY,NIL);
  if (!stream->con || ((sock = SSL_get_fd (stream->con)) < 0)) return NIL;
  (*bn) (BLOCK_TCPREAD,NIL);
  while (stream->ictr < 1) {	/* if nothing in the buffer */
    if ((i = SSL_read (stream->con,stream->ibuf,SSLBUFLEN)) <= 0) {
      time_t tl = time (0);	/* start of request */
      tmo.tv_sec = ttmo_read;	/* read timeout */
      tmo.tv_usec = 0;
      FD_ZERO (&fds);		/* initialize selection vector */
      FD_ZERO (&efds);		/* handle errors too */
      FD_SET (sock,&fds);	/* set bit in selection vector */
      FD_SET (sock,&efds);	/* set bit in error selection vector */
      errno = NIL;		/* block and read */
      while (((i = select (sock+1,&fds,0,&efds,ttmo_read ? &tmo : 0)) < 0)
	     && (errno == EINTR));
      if (!i) {			/* timeout? */
	time_t tc = time (0);
	if (tmoh && ((*tmoh) (tc - t,tc - tl))) continue;
	else return ssl_abort (stream);
      }
      else if (i < 0) return ssl_abort (stream);
      while (((i = SSL_read (stream->con,stream->ibuf,SSLBUFLEN)) < 0) &&
	     (errno == EINTR));
      if (i < 1) return ssl_abort (stream);
    }
    stream->iptr = stream->ibuf;/* point at TCP buffer */
    stream->ictr = i;		/* set new byte count */
  }
  (*bn) (BLOCK_NONE,NIL);
  return T;
}

/* SSL send string as record
 * Accepts: SSL stream
 *	    string pointer
 * Returns: T if success else NIL
 */

long ssl_soutr (SSLSTREAM *stream,char *string)
{
  return ssl_sout (stream,string,(unsigned long) strlen (string));
}


/* SSL send string
 * Accepts: SSL stream
 *	    string pointer
 *	    byte count
 * Returns: T if success else NIL
 */

long ssl_sout (SSLSTREAM *stream,char *string,unsigned long size)
{
  long i;
  extern long maxposint;
  blocknotify_t bn = (blocknotify_t) mail_parameters (NIL,GET_BLOCKNOTIFY,NIL);
  if (!stream->con) return NIL;
  (*bn) (BLOCK_TCPWRITE,NIL);
				/* until request satisfied */
  for (i = 0; size > 0; string += 1,size -= i)
				/* write as much as we can */
    if ((i = SSL_write (stream->con,string,(int) min (maxposint,size))) < 0)
      return ssl_abort (stream);/* write failed */
  (*bn) (BLOCK_NONE,NIL);
  return LONGT;			/* all done */
}

/* SSL close
 * Accepts: SSL stream
 */

void ssl_close (SSLSTREAM *stream)
{
  ssl_abort (stream);		/* nuke the stream */
  fs_give ((void **) &stream);	/* flush the stream */
}


/* SSL abort stream
 * Accepts: SSL stream
 * Returns: NIL always
 */

long ssl_abort (SSLSTREAM *stream)
{
  blocknotify_t bn = (blocknotify_t) mail_parameters (NIL,GET_BLOCKNOTIFY,NIL);
  if (stream->con) {		/* close SSL connection */
    SSL_shutdown (stream->con);
    SSL_free (stream->con);
    stream->con = NIL;
  }
  if (stream->context) {	/* clean up context */
    SSL_CTX_free (stream->context);
    stream->context = NIL;
  }
  if (stream->tcpstream) {	/* close TCP stream */
    tcp_close (stream->tcpstream);
    stream->tcpstream = NIL;
  }
  (*bn) (BLOCK_NONE,NIL);
  return NIL;
}

/* SSL get host name
 * Accepts: SSL stream
 * Returns: host name for this stream
 */

char *ssl_host (SSLSTREAM *stream)
{
  return tcp_host (stream->tcpstream);
}


/* SSL get remote host name
 * Accepts: SSL stream
 * Returns: host name for this stream
 */

char *ssl_remotehost (SSLSTREAM *stream)
{
  return tcp_remotehost (stream->tcpstream);
}


/* SSL return port for this stream
 * Accepts: SSL stream
 * Returns: port number for this stream
 */

unsigned long ssl_port (SSLSTREAM *stream)
{
  return tcp_port (stream->tcpstream);
}


/* SSL get local host name
 * Accepts: SSL stream
 * Returns: local host name
 */

char *ssl_localhost (SSLSTREAM *stream)
{
  return tcp_localhost (stream->tcpstream);
}

/* Client authenticator
 * Accepts: challenger function
 *	    responder function
 *	    parsed network mailbox structure
 *	    stream argument for functions
 *	    pointer to current trial count
 *	    returned user name
 * Returns: T if success, NIL otherwise, number of trials incremented if retry
 */

long auth_plain_client (authchallenge_t challenger,authrespond_t responder,
			NETMBX *mb,void *stream,unsigned long *trial,
			char *user)
{
  char *s,*t,*u,pwd[MAILTMPLEN];
  void *chal;
  unsigned long cl,sl;
  if (!mb->altflag)		/* snarl if not secure session */
    mm_log ("SECURITY PROBLEM: insecure server advertised AUTH=PLAIN",WARN);
				/* get initial (empty) challenge */
  if ((chal = (*challenger) (stream,&cl)) && !cl) {
    fs_give ((void **) &chal);
				/* prompt user */
    mm_login (mb,user,pwd,*trial);
    if (!pwd[0]) {		/* user requested abort */
      (*responder) (stream,NIL,0);
      *trial = 0;		/* don't retry */
      return T;			/* will get a NO response back */
    }
    t = s = (char *) fs_get (sl = strlen (user) + strlen (pwd) + 2);
    *t++ = '\0';		/* use authentication id as authorization id */
				/* copy user name as authentication id */
    for (u = user; *u; *t++ = *u++);
    *t++ = '\0';		/* delimiting NUL */
				/* copy password */
    for (u = pwd; *u; *t++ = *u++);
				/* send credentials */
    if ((*responder) (stream,s,sl) && !(chal = (*challenger) (stream,&cl))) {
      fs_give ((void **) &s);	/* free response */
      ++*trial;			/* can try again if necessary */
      return T;			/* check the authentication */
    }
    fs_give ((void **) &s);	/* free response */
  }
  if (chal) fs_give ((void **) &chal);
  *trial = 0;			/* don't retry */
  return NIL;			/* failed */
}

/* Check if PLAIN valid on this system
 * Returns: T, always
 */

long auth_plain_valid (void)
{
  return T;			/* PLAIN is valid */
}


/* Server authenticator
 * Accepts: responder function
 *	    argument count
 *	    argument vector
 * Returns: authenticated user name or NIL
 */

char *auth_plain_server (authresponse_t responder,int argc,char *argv[])
{
  char *ret = NIL;
  char *user,*aid,*pass;
  unsigned long len;
				/* get user name */
  if (aid = (*responder) ("",0,&len)) {
				/* note: responders null-terminate */
    if ((((unsigned long) ((user = aid + strlen (aid) + 1) - aid)) < len) &&
	(((unsigned long) ((pass = user + strlen (user) + 1) - aid)) < len) &&
	(((unsigned long) ((pass + strlen (pass)) - aid)) == len) &&
	!(*aid && strcmp (aid,user)) && server_login (user,pass,argc,argv))
      ret = myusername ();
    fs_give ((void **) &aid);
  }
  return ret;
}

/* Init server for SSL
 * Accepts: server name
 *	    service name
 *	    alternative service name
 * Returns: SSL stdio stream, always
 */

void server_init (char *server,char *service,char *altservice,char *sasl,
                  void *clkint,void *kodint,void *hupint,void *trmint)
{
  struct servent *sv;
  long port;
  if (server) {			/* set server name in syslog */
    openlog (server,LOG_PID,LOG_MAIL);
    fclose (stderr);		/* possibly save a process ID */
  }
  /* Use SSL if alt service, or if server starts with "s" and not service */
  if (service && altservice && ((port = tcp_serverport ()) >= 0) &&
      (((sv = getservbyname (altservice,"tcp")) &&
	(port == ntohs (sv->s_port))) ||
       ((*server == 's') && (!(sv = getservbyname (service,"tcp")) ||
			     (port != ntohs (sv->s_port))))))
    sslstdio = (void *) ssl_server_init (server);
  else auth_ssl.server = NIL;	/* server forbids PLAIN if not SSL */
				/* now call c-client's version */
  Server_init (NIL,service,altservice,sasl,clkint,kodint,hupint,trmint);
}

				/* link to the real one */
#define server_init Server_init

/* Wait for stdin input
 * Accepts: timeout in seconds
 * Returns: T if have input on stdin, else NIL
 */

long server_input_wait (long seconds)
{
  int i,sock;
  fd_set fds,efd;
  struct timeval tmo;
  SSLSTREAM *stream;
  if (!sslstdio) return Server_input_wait (seconds);
				/* input available in buffer */
  if (((stream = sslstdio->sslstream)->ictr > 0) ||
      !stream->con || ((sock = SSL_get_fd (stream->con)) < 0)) return LONGT;
				/* input available from SSL */
  if ((i = SSL_read (stream->con,stream->ibuf,SSLBUFLEN)) > 0) {
    stream->iptr = stream->ibuf;/* point at TCP buffer */
    stream->ictr = i;		/* set new byte count */
    return LONGT;
  }
  FD_ZERO (&fds);		/* initialize selection vector */
  FD_ZERO (&efd);		/* initialize selection vector */
  FD_SET (sock,&fds);		/* set bit in selection vector */
  FD_SET (sock,&efd);		/* set bit in selection vector */
  tmo.tv_sec = seconds; tmo.tv_usec = 0;
				/* see if input available from the socket */
  return select (sock+1,&fds,0,&efd,&tmo) ? LONGT : NIL;
}

				/* link to the other one */
#define server_input_wait Server_input_wait

/* Init server for SSL
 * Accepts: server name
 * Returns: SSL stdio stream, always
 */

SSLSTDIOSTREAM *ssl_server_init (char *server)
{
  char tmp[MAILTMPLEN];
  unsigned long i;
  struct stat sbuf;
  struct sockaddr_in sin;
  int sinlen = sizeof (struct sockaddr_in);
  SSLSTREAM *stream = (SSLSTREAM *) memset (fs_get (sizeof (SSLSTREAM)),0,
					    sizeof (SSLSTREAM));
  ssl_onceonlyinit ();		/* make sure algorithms added */
  ERR_load_crypto_strings ();
  SSL_load_error_strings ();
				/* get socket address */
  if (getsockname (0,(struct sockaddr *) &sin,(void *) &sinlen))
    fatal ("Impossible getsockname failure!");
				/* build specific certificate/key file name */
  sprintf (tmp,"%s%s-%s.pem",SSL_CERT_DIRECTORY,server,
	   inet_ntoa (sin.sin_addr));
				/* use non-specific name if no specific file */
  if (stat (tmp,&sbuf)) sprintf (tmp,"%s%s.pem",SSL_CERT_DIRECTORY,server);
				/* create context */
  if (stream->context = SSL_CTX_new (SSLv23_server_method ())) {
    SSL_CTX_set_options (stream->context,SSL_OP_ALL);
				/* load certificate */
    if (!SSL_CTX_use_certificate_file (stream->context,tmp,SSL_FILETYPE_PEM))
      syslog (LOG_ALERT,"Unable to load certificate from %s",tmp);
				/* load key */
    else if (!(SSL_CTX_use_RSAPrivateKey_file (stream->context,tmp,
					       SSL_FILETYPE_PEM)))
      syslog (LOG_ALERT,"Unable to load private key from %s",tmp);
    else {			/* generate key if needed */
      if (SSL_CTX_need_tmp_RSA (stream->context))
	SSL_CTX_set_tmp_rsa_callback (stream->context,ssl_genkey);
				/* create new SSL connection */
      if (stream->con = SSL_new (stream->context)) {
				/* set file descriptor */
	SSL_set_fd (stream->con,0);
				/* all OK if accepted */
	if (SSL_accept (stream->con) >= 0) {
	  SSLSTDIOSTREAM *ret = (SSLSTDIOSTREAM *)
	    memset (fs_get (sizeof(SSLSTDIOSTREAM)),0,sizeof(SSLSTDIOSTREAM));
	  ret->sslstream = stream;
	  ret->octr = SSLBUFLEN;/* available space in output buffer */
	  ret->optr = ret->obuf;/* current output buffer pointer */
	  return ret;
	}
      }
    }  
  }
  while (i = ERR_get_error ())	/* SSL failure */
    syslog (LOG_ERR,"SSL error status: %s",ERR_error_string (i,NIL));
  ssl_close (stream);		/* punt stream */
  exit (1);			/* punt this program too */
}

/* Generate one-time key for server
 * Accepts: SSL connection
 *	    export flag
 *	    keylength
 * Returns: generated key, always
 */

static RSA *ssl_genkey (SSL *con,int export,int keylength)
{
  static RSA *key = NIL;
  if (!key &&			/* make sure have a key */
      !(key = RSA_generate_key (export ? keylength : 1024,RSA_F4,NULL,NULL))) {
    unsigned long i;
    syslog (LOG_ALERT,"Unable to generate temp key");
    while (i = ERR_get_error ())
      syslog (LOG_ALERT,"SSL error status: %s",ERR_error_string (i,NIL));
    exit (1);
  }
  return key;
}

/* Get character
 * Returns: character or EOF
 */

int ssl_getchar (void)
{
  if (!sslstdio) return getchar ();
  if (!ssl_getdata (sslstdio->sslstream)) return EOF;
				/* one last byte available */
  sslstdio->sslstream->ictr--;
  return (int) *(sslstdio->sslstream->iptr)++;
}


/* Get string
 * Accepts: destination string pointer
 *	    number of bytes available
 * Returns: destination string pointer or NIL if EOF
 */

char *ssl_gets (char *s,int n)
{
  int i,c;
  if (!sslstdio) return fgets (s,n,stdin);
  for (i = c = 0, n-- ; (c != '\n') && (i < n); sslstdio->sslstream->ictr--) {
    if ((sslstdio->sslstream->ictr <= 0) && !ssl_getdata (sslstdio->sslstream))
      return NIL;		/* read error */
    c = s[i++] = *(sslstdio->sslstream->iptr)++;
  }
  s[i] = '\0';			/* tie off string */
  return s;
}

/* Put character
 * Accepts: character
 * Returns: character written or EOF
 */

int ssl_putchar (int c)
{
  if (!sslstdio) return putchar (c);
				/* flush buffer if full */
  if (!sslstdio->octr && ssl_flush ()) return EOF;
  sslstdio->octr--;		/* count down one character */
  *sslstdio->optr++ = c;	/* write character */
  return c;			/* return that character */
}


/* Put string
 * Accepts: destination string pointer
 * Returns: 0 or EOF if error
 */

int ssl_puts (char *s)
{
  if (!sslstdio) return fputs (s,stdout);
  while (*s) {			/* flush buffer if full */
    if (!sslstdio->octr && ssl_flush ()) return EOF;
    *sslstdio->optr++ = *s++;	/* write one more character */
    sslstdio->octr--;		/* count down one character */
  }
  return 0;			/* success */
}


/* Flush output
 * Returns: 0 or EOF if error
 */

int ssl_flush (void)
{
  if (!sslstdio) return fflush (stdout);
				/* force out buffer */
  if (!ssl_sout (sslstdio->sslstream,sslstdio->obuf,
		 SSLBUFLEN - sslstdio->octr)) return EOF;
				/* renew output buffer */
  sslstdio->optr = sslstdio->obuf;
  sslstdio->octr = SSLBUFLEN;
  return 0;			/* success */
}
